/*
 * Copyright (c) 1997, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.util;

import org.checkerframework.checker.index.qual.NonNegative;
import org.checkerframework.checker.index.qual.PolyGrowShrink;
import org.checkerframework.checker.index.qual.Shrinkable;
import org.checkerframework.checker.lock.qual.GuardSatisfied;
import org.checkerframework.checker.nonempty.qual.EnsuresNonEmpty;
import org.checkerframework.checker.nonempty.qual.EnsuresNonEmptyIf;
import org.checkerframework.checker.nonempty.qual.NonEmpty;
import org.checkerframework.checker.nonempty.qual.PolyNonEmpty;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.checkerframework.checker.nullness.qual.PolyNull;
import org.checkerframework.checker.signedness.qual.PolySigned;
import org.checkerframework.checker.signedness.qual.UnknownSignedness;
import org.checkerframework.dataflow.qual.Pure;
import org.checkerframework.dataflow.qual.SideEffectFree;
import org.checkerframework.framework.qual.AnnotatedFor;
import org.checkerframework.framework.qual.CFComment;
import java.util.function.IntFunction;
import java.util.function.Predicate;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

@CFComment("lock/nullness: Subclasses of this interface/class may opt to prohibit null elements")
@AnnotatedFor({ "lock", "nullness", "index" })
public interface Collection<E> extends Iterable<E> {

    @Pure
    @NonNegative
    int size(@GuardSatisfied Collection<E> this);

    @Pure
    @EnsuresNonEmptyIf(result = false, expression = "this")
    boolean isEmpty(@GuardSatisfied Collection<E> this);

    @CFComment({ "lock: not true, because map could contain nulls:  AssertParametersNonNull(\"get(#1)\")", "Nullness: `o` is not @Nullable because this collection might forbid null", "(though I think a nicer specification would be to return false in that case)" })
    @Pure
    @EnsuresNonEmptyIf(result = true, expression = "this")
    boolean contains(@GuardSatisfied Collection<E> this, @GuardSatisfied @UnknownSignedness Object o);

    @SideEffectFree
    @PolyGrowShrink
    @PolyNonEmpty
    Iterator<E> iterator(@PolyGrowShrink @PolyNonEmpty Collection<E> this);

    @CFComment({ "lock: The Nullness Checker does NOT use these signatures for either version", "of toArray; rather, the checker has hard-coded rules for those two", "methods, because the most useful type for toArray is not expressible", "in the surface syntax that the nullness annotations support." })
    @SideEffectFree
    @PolyNull
    @PolySigned
    Object[] toArray(Collection<@PolyNull @PolySigned E> this);

    @SideEffectFree
    @Nullable
    <T extends @UnknownSignedness Object> T[] toArray(@PolyNull T[] a);

    default <T> T[] toArray(IntFunction<T[]> generator);

    @EnsuresNonEmpty("this")
    boolean add(@GuardSatisfied Collection<E> this, E e);

    boolean remove(@GuardSatisfied @Shrinkable Collection<E> this, @UnknownSignedness Object o);

    @Pure
    boolean containsAll(@GuardSatisfied Collection<E> this, @GuardSatisfied Collection<? extends @UnknownSignedness Object> c);

    boolean addAll(@GuardSatisfied Collection<E> this, Collection<? extends E> c);

    boolean removeAll(@GuardSatisfied @Shrinkable Collection<E> this, Collection<? extends @UnknownSignedness Object> c);

    default boolean removeIf(@Shrinkable Collection<E> this, Predicate<? super E> filter);

    boolean retainAll(@GuardSatisfied @Shrinkable Collection<E> this, Collection<? extends @UnknownSignedness Object> c);

    void clear(@GuardSatisfied @Shrinkable Collection<E> this);

    @Pure
    boolean equals(@GuardSatisfied Collection<E> this, @GuardSatisfied @Nullable @UnknownSignedness Object o);

    @Pure
    int hashCode(@GuardSatisfied Collection<E> this);

    @SideEffectFree
    @Override
    default Spliterator<E> spliterator();

    @PolyNonEmpty
    default Stream<E> stream(@PolyNonEmpty Collection<E> this);

    default Stream<E> parallelStream();
}
